INTEGER, PLUS, MINUS, MUL, DIV, EOF, LPAREN, RPAREN = 'INTEGER', 'PLUS', 'MINUS', 'MUL', 'DIV', 'EOF', 'LPAREN', 'RPAREN'


class Token:
    def __init__(self, value_type, value):
        self.value_type = value_type
        self.value = value

    def __str__(self):
        return 'Token({value_type},{value})'.format(value_type=self.value_type, value=self.value)

    def __repr__(self):
        return self.__str__()


ASSIGN, SEMI, DOT, BEGIN, END, ID = 'ASSIGN', 'SEMI', 'DOT', 'BEGIN', 'END', 'ID'
RESERVED_KEYWORDS = {
    'BEGIN': Token(BEGIN, 'BEGIN'),
    'END': Token(END, 'END'),
    'DIV': Token(DIV, 'DIV')  # 添加整除的保留字实现通过关键字进行整除
}


class Lexer:
    def __init__(self, text):
        self.text = text
        self.position = 0
        self.current_char = self.text[self.position]

    def error(self):
        raise Exception('警告：错误的输入内容！')

    def _id(self):
        result = ''
        if self.current_char == '_':  # 此语句块实现变量名下划线开头
            result = '_'
            self.advance()
        while self.current_char is not None and self.current_char.isalnum():
            result += self.current_char
            self.advance()
        token = RESERVED_KEYWORDS.get(result.upper(), Token('ID', result.lower()))  # 此语句块实现忽略关键字和变量名大小写
        return token

    def peek(self):
        pos = self.position + 1
        if pos >= len(self.text):
            return None
        else:
            return self.text[pos]

    def skip_whitespace(self):
        while self.current_char is not None and self.current_char.isspace():
            self.advance()

    def long_integer(self):
        result = ''
        while self.current_char is not None and self.current_char.isdigit():
            result += self.current_char
            self.advance()
        return int(result)

    def advance(self):
        self.position += 1
        if self.position >= len(self.text):
            self.current_char = None
        else:
            self.current_char = self.text[self.position]

    def get_next_token(self):
        while self.current_char is not None:
            if self.current_char.isspace():
                self.skip_whitespace()
                continue
            if self.current_char.isalpha() or self.current_char == '_':  # 如果当前字符是字母或下划线（支持变量名称下划线开头）
                return self._id()
            if self.current_char == ':' and self.peek() == '=':
                self.advance()
                self.advance()
                return Token(ASSIGN, ':=')
            if self.current_char == ';':
                self.advance()
                return Token(SEMI, ';')
            if self.current_char == '.':
                self.advance()
                return Token(DOT, '.')
            if self.current_char.isdigit():
                return Token(INTEGER, self.long_integer())
            if self.current_char == '+':
                self.advance()
                return Token(PLUS, '+')
            if self.current_char == '-':
                self.advance()
                return Token(MINUS, '-')
            if self.current_char == '*':
                self.advance()
                return Token(MUL, '*')
            # 此处删除了斜杠表示整除的语句
            if self.current_char == '(':
                self.advance()
                return Token(LPAREN, '(')
            if self.current_char == ')':
                self.advance()
                return Token(RPAREN, ')')
            self.error()
        return Token(EOF, None)


class AST:
    pass


class Compound(AST):
    def __init__(self):
        self.children = []


class Assign(AST):
    def __init__(self, left, operator, right):
        self.left = left
        self.token = self.operator = operator
        self.right = right


class Variable(AST):
    def __init__(self, token):
        self.token = token
        self.name = token.value


class NoOperator(AST):
    pass


class BinaryOperator(AST):
    def __init__(self, left, operator, right):
        self.left = left
        self.token = self.operator = operator
        self.right = right


class Num(AST):
    def __init__(self, token):
        self.token = token
        self.value = token.value


class UnaryOperator(AST):
    def __init__(self, operator, expr):
        self.token = self.operator = operator
        self.expr = expr


class Parser:
    def __init__(self, lexer):
        self.lexer = lexer
        self.current_token = lexer.get_next_token()

    def error(self):
        raise Exception('警告：错误的输入内容！')

    def eat(self, value_type):
        if self.current_token.value_type == value_type:
            self.current_token = self.lexer.get_next_token()
        else:
            self.error()

    def program(self):
        node = self.compound_statement()
        self.eat(DOT)
        return node

    def compound_statement(self):
        self.eat(BEGIN)
        nodes = self.statement_list()
        self.eat(END)

        root = Compound()
        root.children = nodes
        return root

    def statement_list(self):
        node = self.statement()
        nodes = [node]
        while self.current_token.value_type == SEMI:
            self.eat(SEMI)
            nodes.append(self.statement())
        if self.current_token.value_type == ID:
            self.error()
        return nodes

    def statement(self):
        if self.current_token.value_type == BEGIN:
            node = self.compound_statement()
        elif self.current_token.value_type == ID:
            node = self.assignment_statement()
        else:
            node = self.empty()
        return node

    def assignment_statement(self):
        left = self.variable()
        token = self.current_token
        self.eat(ASSIGN)
        right = self.expr()
        node = Assign(left, token, right)
        return node

    def empty(self):
        return NoOperator()

    def variable(self):
        node = Variable(self.current_token)
        self.eat(ID)
        return node

    def factor(self):
        current_token = self.current_token

        if current_token.value_type == PLUS:
            self.eat(PLUS)
            node = UnaryOperator(current_token, self.factor())
            return node
        elif current_token.value_type == MINUS:
            self.eat(MINUS)
            node = UnaryOperator(current_token, self.factor())
            return node
        elif current_token.value_type == INTEGER:
            self.eat(INTEGER)
            return Num(current_token)
        elif current_token.value_type == LPAREN:
            self.eat(LPAREN)
            node = self.expr()
            self.eat(RPAREN)
            return node
        else:
            node = self.variable()
            return node

    def term(self):
        node = self.factor()
        while self.current_token.value_type in (MUL, DIV):
            token = self.current_token
            if token.value_type == MUL:
                self.eat(MUL)
            elif token.value_type == DIV:
                self.eat(DIV)
            node = BinaryOperator(node, token, self.factor())
        return node

    def expr(self):
        node = self.term()
        while self.current_token.value_type in (PLUS, MINUS):
            token = self.current_token
            if token.value_type == PLUS:
                self.eat(PLUS)
            elif token.value_type == MINUS:
                self.eat(MINUS)
            node = BinaryOperator(node, token, self.term())
        return node

    def parser(self):
        node = self.program()
        if self.current_token.value_type != EOF:
            self.error()
        return node


class NodeVisitor:
    def visit(self, node):
        method_name = 'visit_' + type(node).__name__
        visitor = getattr(self, method_name, self.generic_visit)
        return visitor(node)

    def generic_visit(self, node):
        raise Exception(f'未找到“visit_{type(node).__name__}()”方法！')


class Interpreter(NodeVisitor):
    GLOBAL_SCOPE = {}

    def __init__(self, parser):
        self.parser = parser

    def visit_Compound(self, node):
        for child in node.children:
            self.visit(child)

    def visit_Assign(self, node):
        var_name = node.left.name
        self.GLOBAL_SCOPE[var_name] = self.visit(node.right)

    def visit_Variable(self, node):
        var_name = node.name
        value = self.GLOBAL_SCOPE.get(var_name)
        if value is None:
            raise NameError(f'错误的标识符：{repr(var_name)}')
        else:
            return value

    def visit_NoOperator(self, node):
        pass

    def visit_BinaryOperator(self, node):
        if node.operator.value_type == PLUS:
            return self.visit(node.left) + self.visit(node.right)
        elif node.operator.value_type == MINUS:
            return self.visit(node.left) - self.visit(node.right)
        elif node.operator.value_type == MUL:
            return self.visit(node.left) * self.visit(node.right)
        elif node.operator.value_type == DIV:
            return self.visit(node.left) // self.visit(node.right)

    def visit_Num(self, node):
        return node.value

    def visit_UnaryOperator(self, node):
        if node.operator.value_type == PLUS:
            return +self.visit(node.expr)
        if node.operator.value_type == MINUS:
            return -self.visit(node.expr)

    def interpret(self):
        tree = self.parser.parser()
        return self.visit(tree)


def main():
    text = '''
    BEGIN
    
        BEGIN
            number := 2;
            a := NumBer;
            B := 10 * a + 10 * NUMBER DIV 4;
            _c := a - - b
        end;
    
        x := 11;
    END.
    '''
    lexer = Lexer(text)
    parser = Parser(lexer)
    interpreter = Interpreter(parser)
    interpreter.interpret()
    print(interpreter.GLOBAL_SCOPE)


if __name__ == '__main__':
    main()
